#include <windows.h>
#include <gl/gl.h>
/******************************************************************************
 * sphere.cpp - Gestion de la sphere                                          *
 ******************************************************************************
 * Projet       : Code(R) - Sphere                                            *
 * Auteur       : Arnaud Storq (http://norecess.planet-d.net)/Posse-Press 2004*
 ******************************************************************************/
#include "main.h"
#include "sphere.h"
#include "libtarga.h"
#include "math.h"

//---------------------------------------------------------------------------- CONSTANTES LOCALES
#define PI 3.141592654f

//---------------------------------------------------------------------------- VARIABLES LOCALES
static int m_largeur = 0;                       // largeur de l'image en pixels
static int m_hauteur = 0;                       // hauteur de l'image en pixels
static unsigned int m_textureOpenGL;            // identifiant OpenGL de la texture

//---------------------------------------------------------------------------- INITIALISATION
void sphereInitialiser( )
{
    // Chargement de la texture en mmoire
    unsigned char *texture = (unsigned char *) tga_load( FICHIER_TEXTURE, &m_largeur, &m_hauteur, TGA_TRUECOLOR_24 );
    
    // Demander  OpenGL un nouvel identifiant de texture
    glGenTextures( 1, &m_textureOpenGL );

    // Selectionner la nouvelle texture
    glBindTexture( GL_TEXTURE_2D, m_textureOpenGL );

    // Forcer l'affichage bilineaire sur la texture
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );    
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
	
    // Copier la texture prsente en mmoire centrale vers celle de la carte 3D !
    // On doit indiquer  OpenGL la taille de la texture : m_largeur et m_hauteur
    // GL_RGB signifie que la texture est au format RGB 24 bits (8 bits par composantes (R=rouge, G=vert, B=bleu))
    // c'est pour cela que l'on a charg la texture avec le parametre TGA_TRUECOLOR_24
    glTexImage2D( GL_TEXTURE_2D, 0, 3, m_largeur, m_hauteur, 0, GL_RGB, GL_UNSIGNED_BYTE, texture );
    
    // On peut ds  prsent effacer la texture de la mmoire centrale..
    delete [] texture;
}

//---------------------------------------------------------------------------- AFFICHAGE
void sphereAfficher( int time )
{
    // Indiquer la couleur d'effacage de l'ecran avec du noir (R=0, G=0, B=0)
    glClearColor( 0.0f, 0.0f, 0.0f, 0.0f );
    
    // Effacer l'ecran et le Z-Buffer (tampon des Z)
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
            
    // On sauvegarde la matrice actuelle
    glPushMatrix();
            
    // Dimensionne l'ecran OpenGL  la taille de la texture de rendu
    // du coup, glVertex3f(-1.0f, -1.0f, -1.0f) sera plac en haut a gauche
    // et glVertex3f(1.0f, -.0f, -1.0f) sera plac en bas a droite !
    glViewport( 0, 0, RESOLUTION_RENDERTEXTURE_LARGEUR, RESOLUTION_RENDERTEXTURE_HAUTEUR );
            
    // L'animation de la sphere est base sur le temps courrant specifi en parametres
    float temps = SPHERE_ROTATION_VITESSE * (float) time ;
    
    // Applique un leger deplacement de la sphere sur les axes verticaux et horizontaux
    // Une courbe Sinus et Cosinus sont utilises pour donner la sensation de rebondissement dans les airs..
    glTranslatef( SPHERE_MOUVEMENT_X * sinf( temps * 0.05f ),
                  SPHERE_MOUVEMENT_Y * cosf( temps * 0.02f ),
                  -1.0f );
            
    // Applique une rotation sur la sphere, toujours bas sur le temps actuel !
    glRotatef( temps, 1.0f, 0.0f, 0.0f );          // axe X
    glRotatef( 90.0f+temps, 0.0f, 1.0f, 0.0f );    // axe Y
    glRotatef( -temps, 0.0f, 0.0f, 1.0f );         // axe Z
            
    // A prsent que notre rendu est initialis, on peut afficher notre sphere..
    
    // Selectionne la texture destin  etre appliqu sur la sphere
    glBindTexture( GL_TEXTURE_2D, m_textureOpenGL );
    
    // Autorise l'affichage d'une texture
    glEnable( GL_TEXTURE_2D );

    // Autorise le Z-Buffer, cela nous evitant de devoir trier les faces en Z !
    glEnable( GL_DEPTH_TEST );
    
    // Lors de l'affichage d'un triangle, les Z de ce dernier sont compares au Z-Buffer
    // Si le Z courrant est plus petit que celui present dans le Z-Buffer (GL_LESS),
    // alors on l'affiche !
    glDepthFunc( GL_LESS );

    // Calcul des increments, nous permettant de positionner le point courrant par rapport a la sphere
    float horizontalIncrement = PI / SPHERE_NB_POINTS_HORIZONTAL;
    float verticalIncrement = 2.0f * ( PI / SPHERE_NB_POINTS_VERTICAL );
    
    // Pour chaque points horizontaux..
    for ( int i = 0; i < SPHERE_NB_POINTS_HORIZONTAL; ++i )
    {
	    // Calcule la position point courrant sur le cercle horizontal
        float pointCourrant = ( ( float ) i ) * horizontalIncrement;

	    // Calcule la position point suivant sur le cercle horizontal
	    // Il suffit de prendre le point courrant et de lui rajouter l'increment horozontal !
	    float pointSuivant = pointCourrant + horizontalIncrement;
	    
	    // Calcule les deplacements sur l'axe des X
        // On en profite pour multiplier cette valeur par la taille de la sphere 
        float pointCourrantX = SPHERE_TAILLE * sinf( pointCourrant );
	    float pointSuivantX = SPHERE_TAILLE * sinf( pointSuivant );
	    
	    // Calcule les deplacements sur l'axe des Z
        // On en profite pour multiplier cette valeur par la taille de la sphere 
        float pointCourrantZ = SPHERE_TAILLE * cosf( pointCourrant );
	    float pointSuivantZ = SPHERE_TAILLE * cosf( pointSuivant );
	
 	    // Indique que l'on veut afficher des triangles STRIPPs !!!
 	    // Ils sont relis les uns aux autres, dans cet ordre :
 	    // Les 3 premiers points specifis par glVertex3f traceront le 1er triangle,
 	    // et les points suivants re-utiliseront les deux derniers du triangle specifi
 	    // auparavant.. l'avantage est double : non seulement on economise de la bande
 	    // passante (moins de polygones  envoyer a la carte 3D), mais en plus, 
 	    // le code est plus simple ainsi :)
        glBegin( GL_TRIANGLE_STRIP );

 	    // Pour chaque points verticaux..
        for ( int j = 0; j <= SPHERE_NB_POINTS_VERTICAL; ++j )
        {
     	    // Calcule les deplacements sur l'axe des Y
            float courrant = ( ( float ) j ) * verticalIncrement;            
            float x = cosf( courrant );
            float y = sinf( courrant );

            // Specifie la coordonne texture du point courrant
            glTexCoord2f( ( float ) j / (float) SPHERE_NB_POINTS_VERTICAL, ( float ) i / ( float ) SPHERE_NB_POINTS_HORIZONTAL );	    
	        
            // Specifie la position geometrique du point courrant
            glVertex3f( x * pointCourrantX, y * pointCourrantX, pointCourrantZ );

            // Specifie la coordonne texture du point suivant
	        glTexCoord2f( ( float ) j / ( float ) SPHERE_NB_POINTS_VERTICAL, ( i + 1 ) / ( float ) SPHERE_NB_POINTS_HORIZONTAL );

            // Specifie la position geometrique du point suivant
	        glVertex3f( x * pointSuivantX, y * pointSuivantX, pointSuivantZ );
        }	        

        // Indique que l'on a fini d'envoyer des points  la carte 3D !
        glEnd();
    }
    
    // On recupere la matrice precedemment charge !
    glPopMatrix ();
}

//---------------------------------------------------------------------------- DESTRUCTION
void sphereDetruire()
{
    // Indique  OpenGL d'effacer la texture de la carte 3D, plus ses initilisations internes
    glDeleteTextures(1, &m_textureOpenGL);
}
